# Mobile Chat API - PHP Backend

A complete REST API backend for a mobile chat application with message queuing, delivery acknowledgments, and push notifications.

## Features

- ✅ Message sending and receiving
- ✅ Offline message queuing
- ✅ Delivery and read receipts
- ✅ Push notifications (Firebase FCM)
- ✅ Device token management
- ✅ Message expiration (30 days default)
- ✅ CORS enabled for mobile apps
- ✅ Polling-based message delivery

## Directory Structure

```
mobileapi/
├── config/
│   └── database.php          # Database connection configuration
├── api/
│   ├── send_message.php      # Send message endpoint
│   ├── get_messages.php      # Get pending messages endpoint
│   ├── update_status.php     # Update message status endpoint
│   └── register_device.php   # Register device token endpoint
├── cron/
│   └── cleanup.php           # Daily cleanup for expired messages
├── .htaccess                 # URL rewriting rules
├── database_schema.sql       # Database schema
└── README.md                 # This file
```

## Installation

### 1. Database Setup

Create your MySQL database and import the schema:

```bash
mysql -u your_username -p
```

```sql
CREATE DATABASE your_username_chat CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE your_username_chat;
SOURCE database_schema.sql;
```

### 2. Configure Database Connection

Edit `config/database.php` and update the database credentials:

```php
private $host = "localhost";
private $db_name = "your_username_chat";     // Your database name
private $username = "your_db_username";       // Your database username
private $password = "your_db_password";       // Your database password
```

### 3. Enable URL Rewriting

Make sure Apache's `mod_rewrite` is enabled. The `.htaccess` file handles URL routing.

For cPanel hosting, this is typically already enabled.

### 4. Configure Firebase Cloud Messaging (Optional)

If you want push notifications, update the FCM server key in `api/send_message.php`:

```php
$serverKey = 'YOUR_FCM_SERVER_KEY'; // Get from Firebase Console
```

To get your FCM server key:
1. Go to [Firebase Console](https://console.firebase.google.com/)
2. Select your project
3. Go to Project Settings > Cloud Messaging
4. Copy the Server Key

### 5. Set Up Cron Job for Cleanup

In cPanel or your hosting control panel, add a daily cron job:

```bash
0 2 * * * php /path/to/your/mobileapi/cron/cleanup.php
```

This runs daily at 2 AM to clean up expired messages.

## API Endpoints

### Base URL
```
https://yourdomain.com/mobileapi/
```

### 1. Send Message
**POST** `/api/send`

**Request Body:**
```json
{
    "local_message_id": "unique-local-id",
    "sender_id": 1,
    "recipient_id": 2,
    "conversation_id": "conv_1_2",
    "content": "Hello there!",
    "message_type": "text",
    "timestamp": 1698765432000
}
```

**Response:**
```json
{
    "success": true,
    "message": "Message sent successfully",
    "data": {
        "local_message_id": "unique-local-id",
        "server_message_id": 123,
        "status": "sent",
        "timestamp": 1698765432000,
        "recipient_online": false
    }
}
```

### 2. Get Pending Messages
**GET** `/api/messages?user_id={user_id}`

**Response:**
```json
{
    "success": true,
    "count": 2,
    "messages": [
        {
            "server_message_id": 123,
            "local_message_id": "unique-local-id",
            "sender_id": 1,
            "conversation_id": "conv_1_2",
            "content": "Hello there!",
            "message_type": "text",
            "timestamp": 1698765432000,
            "is_edited": false
        }
    ]
}
```

### 3. Update Message Status
**POST** `/api/status`

**Request Body:**
```json
{
    "server_message_id": 123,
    "user_id": 2,
    "status": "delivered"
}
```

Status options: `delivered`, `read`

**Response:**
```json
{
    "success": true,
    "message": "Status updated",
    "data": {
        "server_message_id": 123,
        "status": "delivered",
        "sender_id": 1,
        "timestamp": 1698765432000
    }
}
```

### 4. Register Device Token
**POST** `/api/register`

**Request Body:**
```json
{
    "user_id": 1,
    "device_token": "firebase-device-token",
    "platform": "android"
}
```

**Response:**
```json
{
    "success": true,
    "message": "Device registered successfully"
}
```

## Message Flow

1. **Sending a Message:**
   - App sends message to `/api/send`
   - Server stores message in database
   - If recipient is offline, message is added to queue
   - Push notification sent if device token exists

2. **Receiving Messages:**
   - App polls `/api/messages?user_id={id}` every 5 seconds
   - Server returns all pending messages from queue
   - App sends delivery acknowledgment to `/api/status`
   - Message removed from queue

3. **Read Receipts:**
   - When user opens/reads message, app sends status update
   - Sender can poll for status changes or use WebSocket (future enhancement)

## Android Integration Example

### Retrofit Interface
```java
public interface ApiService {
    @POST("api/send")
    Call<SendMessageResponse> sendMessage(@Body SendMessageRequest request);
    
    @GET("api/messages")
    Call<GetMessagesResponse> getPendingMessages(@Query("user_id") int userId);
    
    @POST("api/status")
    Call<StatusUpdateResponse> updateMessageStatus(@Body StatusUpdateRequest request);
    
    @POST("api/register")
    Call<RegisterDeviceResponse> registerDevice(@Body RegisterDeviceRequest request);
}
```

### Retrofit Client Setup
```java
public class RetrofitClient {
    private static final String BASE_URL = "https://yourdomain.com/mobileapi/";
    private static Retrofit retrofit = null;
    
    public static Retrofit getClient() {
        if (retrofit == null) {
            OkHttpClient client = new OkHttpClient.Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .build();
            
            retrofit = new Retrofit.Builder()
                .baseUrl(BASE_URL)
                .client(client)
                .addConverterFactory(GsonConverterFactory.create())
                .build();
        }
        return retrofit;
    }
}
```

## Security Considerations

### Current Implementation
This is a basic implementation. For production, you should add:

1. **Authentication & Authorization**
   - Add JWT or session-based authentication
   - Validate user permissions before operations
   - Add API keys or OAuth

2. **Input Validation**
   - Sanitize all user inputs
   - Validate message length limits
   - Check for SQL injection (already using PDO prepared statements)

3. **Rate Limiting**
   - Limit API calls per user/IP
   - Prevent spam and abuse

4. **HTTPS**
   - Always use HTTPS in production
   - Update `.htaccess` to force HTTPS:
   ```apache
   RewriteCond %{HTTPS} off
   RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
   ```

5. **Error Handling**
   - Don't expose database errors in production
   - Log errors securely
   - Return generic error messages to clients

## Performance Optimization

### For High Traffic
1. **Add Database Indexing** (already included in schema)
2. **Implement Caching**
   - Use Redis or Memcached for active conversations
3. **Use WebSockets**
   - Replace polling with WebSocket connections
   - Implement using Socket.io or similar
4. **Message Pagination**
   - Add pagination to get_messages.php
   - Limit results per request

### Database Optimization
```sql
-- Add indexes for better performance (already in schema)
CREATE INDEX idx_conversation ON messages(conversation_id);
CREATE INDEX idx_recipient ON message_queue(recipient_id);
```

## Troubleshooting

### Common Issues

1. **"Connection error" message**
   - Check database credentials in `config/database.php`
   - Verify database exists and user has permissions

2. **404 errors on API endpoints**
   - Ensure `mod_rewrite` is enabled
   - Check `.htaccess` file is in root directory
   - Verify file permissions

3. **CORS errors from mobile app**
   - Check CORS headers in `.htaccess`
   - May need to adjust allowed origins for security

4. **Messages not being queued**
   - Check `users` table has `is_online` field
   - Verify `message_queue` table exists

## Testing

### Using cURL

```bash
# Send a message
curl -X POST https://yourdomain.com/mobileapi/api/send \
  -H "Content-Type: application/json" \
  -d '{
    "local_message_id": "test-123",
    "sender_id": 1,
    "recipient_id": 2,
    "conversation_id": "conv_1_2",
    "content": "Test message",
    "message_type": "text",
    "timestamp": 1698765432000
  }'

# Get pending messages
curl -X GET "https://yourdomain.com/mobileapi/api/messages?user_id=2"

# Update status
curl -X POST https://yourdomain.com/mobileapi/api/status \
  -H "Content-Type: application/json" \
  -d '{
    "server_message_id": 123,
    "user_id": 2,
    "status": "delivered"
  }'
```

### Using Postman
1. Import the endpoints as a collection
2. Set base URL: `https://yourdomain.com/mobileapi/`
3. Test each endpoint with sample data

## Future Enhancements

- [ ] WebSocket support for real-time messaging
- [ ] Group chat functionality
- [ ] Message encryption (end-to-end)
- [ ] File upload support with cloud storage
- [ ] Voice messages
- [ ] Video calls integration
- [ ] User blocking and reporting
- [ ] Message search functionality
- [ ] Typing indicators
- [ ] Online presence updates

## License

This project is provided as-is for educational and commercial purposes.

## Support

For issues or questions, please review the code comments and this documentation.

---

**Last Updated:** October 2025
**Version:** 1.0.0

