<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

include_once '../config/database.php';

// Get posted data
$data = json_decode(file_get_contents("php://input"));

// Validate required fields
if (
    !empty($data->local_message_id) &&
    !empty($data->sender_id) &&
    !empty($data->recipient_id) &&
    !empty($data->conversation_id) &&
    !empty($data->content)
) {
    
    $database = new Database();
    $db = $database->getConnection();
    
    try {
        // Insert message
        $query = "INSERT INTO messages 
                  (local_message_id, sender_id, recipient_id, conversation_id, 
                   content, message_type, timestamp, status, sent_at)
                  VALUES 
                  (:local_id, :sender, :recipient, :conversation, 
                   :content, :type, :timestamp, 'sent', NOW())";
        
        $stmt = $db->prepare($query);
        
        $stmt->bindParam(":local_id", $data->local_message_id);
        $stmt->bindParam(":sender", $data->sender_id);
        $stmt->bindParam(":recipient", $data->recipient_id);
        $stmt->bindParam(":conversation", $data->conversation_id);
        $stmt->bindParam(":content", $data->content);
        $stmt->bindParam(":type", $data->message_type);
        $stmt->bindParam(":timestamp", $data->timestamp);
        
        if ($stmt->execute()) {
            $server_message_id = $db->lastInsertId();
            
            // Check if recipient is online
            $check_query = "SELECT is_online, device_token FROM users WHERE id = :recipient_id";
            $check_stmt = $db->prepare($check_query);
            $check_stmt->bindParam(":recipient_id", $data->recipient_id);
            $check_stmt->execute();
            $recipient = $check_stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($recipient && !$recipient['is_online']) {
                // Recipient is offline - add to queue
                $queue_query = "INSERT INTO message_queue (message_id, recipient_id) 
                               VALUES (:msg_id, :recipient_id)";
                $queue_stmt = $db->prepare($queue_query);
                $queue_stmt->bindParam(":msg_id", $server_message_id);
                $queue_stmt->bindParam(":recipient_id", $data->recipient_id);
                $queue_stmt->execute();
                
                // Send push notification if device token exists
                if (!empty($recipient['device_token'])) {
                    sendPushNotification(
                        $recipient['device_token'], 
                        $data->content,
                        $data->sender_id
                    );
                }
            }
            
            // Return success response
            http_response_code(201);
            echo json_encode([
                "success" => true,
                "message" => "Message sent successfully",
                "data" => [
                    "local_message_id" => $data->local_message_id,
                    "server_message_id" => $server_message_id,
                    "status" => "sent",
                    "timestamp" => time() * 1000,
                    "recipient_online" => $recipient['is_online'] ?? false
                ]
            ]);
        } else {
            http_response_code(503);
            echo json_encode([
                "success" => false,
                "message" => "Unable to send message"
            ]);
        }
        
    } catch(PDOException $e) {
        http_response_code(500);
        echo json_encode([
            "success" => false,
            "message" => "Database error: " . $e->getMessage()
        ]);
    }
    
} else {
    http_response_code(400);
    echo json_encode([
        "success" => false,
        "message" => "Missing required fields"
    ]);
}

// Push notification function (using Firebase Cloud Messaging)
function sendPushNotification($deviceToken, $message, $senderId) {
    // Firebase Cloud Messaging implementation
    // You'll need to set up FCM and add your server key
    
    $serverKey = 'YOUR_FCM_SERVER_KEY'; // Get from Firebase Console
    
    $notification = [
        'title' => 'New Message',
        'body' => $message,
        'sound' => 'default',
        'badge' => 1
    ];
    
    $data = [
        'sender_id' => $senderId,
        'type' => 'new_message'
    ];
    
    $fcmMsg = [
        'to' => $deviceToken,
        'notification' => $notification,
        'data' => $data,
        'priority' => 'high'
    ];
    
    $headers = [
        'Authorization: key=' . $serverKey,
        'Content-Type: application/json'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://fcm.googleapis.com/fcm/send');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fcmMsg));
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    return $result;
}
?>

